function[dataNet] =  scaleNet(dataNet)
% Scales the training input and output data for the Matlab LM algorithm.  
% Creates training and test input and output vectors for the Pyrenn LM
% algorithm but does not scale any of the data.
%
% scaleNet = scale the data for Neural Network training and test

% Load data from the data structure.

% Input x-axis scale factor.
sIn 			= dataNet.sIn;
% Output y-axis scale factor.
sOut 			= dataNet.sOut;
% Plot variable option for displaying the plot of scaled data.
sPlot			= dataNet.sPlot;
% Training x-axis data array.
x				= dataNet.x;
% Training y-axis data array.
y 				= dataNet.y;
% Test x-axis data arrays.
xt 			= dataNet.xt;
% Test y-axis data arrays.
yt 			= dataNet.yt;

%--------------------------------------------------------------------------
% Scale training inputs to outputs using Y = M*X + B,
% where M = slope, and B = y-axis intercept.

% Set the x-axis input bounds.
x1 = min(xt);
x2 = max(xt);

% Set the y-axis output bounds.
y1 = -sIn;
y2 = sIn;  

% Compute the slope.
m_in = (y2-y1) / (x2-x1);
% Compute the y-axis intercept.
b_in = y1 - (m_in * x1); 

% Build the Neural Network x-axis input training set scaled from 
% the x-axis input training data.
nnet_in = m_in * x + b_in;

%--------------------------------------------------------------------------
% Scale training output using Y = M*X + B,
% where M = slope, and B = y-axis intercept.

clear x1 x2 y1 y2;

% Set the x-axis input bounds.
x1 = 0.0; % Set this to zero ??
x2 = max(yt);

% Set the y-axis output bounds.
y1 = -sOut;
y2 = sOut;  

% Compute the slope.
m_out = (y2-y1) / (x2-x1);
% Compute the intercept.
b_out = y1 - (m_out * x1); 

% Build the Neural Network x-axis input training set scaled from the 
% y-axis output training data.
nnet_out = m_out * y + b_out;

% Build the Neural Network x-axis input test set scaled from 
% the input test data.
ntest_in = m_in * xt + b_in;

% Build the Neural Network y-axis output test set scaled from the 
% output test data.
ntest_out = m_out * yt + b_out;

%--------------------------------------------------------------------------
% Plot the scaled inputs and outputs.

if sPlot == 1
    
    %----------------------------------------------------------------------
    % Input Data
   
    size1 = size(nnet_in);
    size2 = size(ntest_in);
   
    figure(1);
    
    plot(1:size1(2),nnet_in,'b*',1:size2(2),ntest_in,'r*');
    hold on;
	plot(1:size1(2),nnet_in,'b-',1:size2(2),ntest_in,'r-');   
    grid on;
    xlabel('Training and Test Input Numbers');
    ylabel('Values of Input Training and Test Points');
    legend('Training Data','Test Data');
    title('Scaled Input Training & Test Data');
   
    %----------------------------------------------------------------------
    % Output Data
   
    size1 = size(nnet_out);
    size2 = size(ntest_out);
   
    figure(2);
   
    plot(1:size1(2),nnet_out,'b*',1:size2(2),ntest_out,'r*');
    hold on;
    plot(1:size1(2),nnet_out,'b-',1:size2(2),ntest_out,'r-');
    grid on;
    xlabel('Training and Test Output Numbers');
    ylabel('Values of Output Training and Test Points');
    legend('Training Data','Test Data');
    title('Scaled Output Training & Test Data');
   
    pause;
   
    close all;
   
end

% Load data structure with pertinent values.

% Neural Network x-axis input training set
dataNet.nnet_in 		= nnet_in;
% Neural Network y-axis output training set
dataNet.nnet_out		= nnet_out;
% Neural Network x-axis input test set
dataNet.ntest_in		= ntest_in;
% Neural Network y-axis output test set
dataNet.ntest_out		= ntest_out;
% Neural Network output scale factor - slope
dataNet.m_out			= m_out;
% Neural Network output scale factor - intercept
dataNet.b_out			= b_out;

end