
package earthproj;

import gov.nasa.worldwind.awt.WorldWindowGLCanvas;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.layers.RenderableLayer;
import gov.nasa.worldwind.render.BasicShapeAttributes;
import gov.nasa.worldwind.render.Material;
import gov.nasa.worldwind.render.Path;
import gov.nasa.worldwind.render.Polyline;

import java.awt.Color;
import java.util.ArrayList;

/**
 * This class contains all of the properties and behavior (methods) of a 
 * trajectory object - one which moves on the surface of the Earth, through the
 * atmosphere, or in space.
 * 
 * @author mikescodeprojects
 */
public class TrajectoryObjects
{
    // Trajectory Object properties.  These consist of two different type of
    // line / curve generators: 1) Polyline (deprecated), and 2) Path.  The
    // ArrayList is used to store the position elements (from class Position).
    private Polyline polyline;
    private Path path;
    private final ArrayList<Position> pos;
    // Polyline properties
    //      Use the Path line / curve generator.
    private boolean usePath;
    //      Use the Polyline line / curve generator.
    private boolean usePolyline;

    // WorldWind Earth model properties.
    private WorldWindowGLCanvas wwd;
    private final RenderableLayer layer;

    // Trajectory propagation properties
    private int ict;
    private int tct;
    //      Variables for building the trajectory propagator.
    private double Amp;
    private double rangeFact;
    private double latRate;
    private double lonRate;
    private double resDial;
    //      Number of points in the generated trajectory.
    private int numPoints;
    //      Status which indicates if trajectory is still being propagated.
    private boolean trajRun;

    // Trajectory properties
    //      Current coordinates in the trajectory
    private double lat;
    private double lon;
    private double alt;
    //      Previous coordinates in the trajectory
    private double latPrev;
    private double lonPrev;
    private double altPrev;

    /**
     * The ArrayList and RenderableLayer objects are constructed.  The counters
     * ict and tct are initialized and the boolean trajRun is set to true.
     * 
     */
    public TrajectoryObjects()
    {
        this.ict    = 1;

        // Trajectory status - if true then object is still active in then
        // trajectory, otherwise the trajectory is complete.
        trajRun     = true;

        // ArrayList which contains the object type Polyline or Path - this is
        // inserted into a Layer object for showing the trajectory propgation
        // in the Earth model.
        pos         = new ArrayList <>();
        // The Earth Layer object.
        layer       = new RenderableLayer();
        // The counter for the ArrayList which holds the Position object.  Only
        // cycles between 0 and 1 (reset each time).
        ict         = 0;
        // Simulation counter - it is incremented until the number of simulation
        // points has been exceeded.
        tct         = 0;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to propagate the trajectory through one time step.  
     * The previous and current positions are loaded into the ArrayList and then 
     * into either the Polyline or Path object.  Then the Earth model rendering 
     * is updated.
     * 
     * @return The status of the trajectory: 1) true = still active - trajectory
     * is not complete, or 2) false = inactive - trajectory is complete.
     */
    public boolean propagateTrajectory()
    {
        // If the index is past the maximum number of points or the altitude
        // is less than zero, then mark the object as having completed the
        // trajectory.
        if( tct > numPoints || alt < 0.0)
        {
            trajRun = false;
            // Don't do anything else since the trajectory is complete - just
            // exit.
            return trajRun;
        }

        // Load 1st element (position of the trajectory object) 
        // into the ArrayList.
        pos.add(ict, Position.fromDegrees(latPrev, lonPrev, altPrev));
        // Increment the ArrayList counter for the next Position object.
        ict++;

        // Increment surface coordinates.
        lat = lat + latRate;
        lon = lon + lonRate;
        // Increment altitude coordinate.
        alt = Amp * Math.sin( rangeFact * ( Math.toRadians(tct)) );
        // Load 2nd element (updated position of the trajector object) 
        // into the ArrayList.
        pos.add(ict,Position.fromDegrees(lat,lon,alt));

        // If "true" then use the Polyline class to build the line / curve.
        if( usePolyline )
        {
            polyline.setPositions(pos);
        }
        // Otherwise, use the Path class to build the line / curve.
        else if( usePath )
        {
            path.setPositions(pos);
        }

        // Update the graphics rendering of the Earth model.
        wwd.redrawNow();

        // The current coordinates are now the "previous" coordinates before
        // proceeding to the next update.
        latPrev = lat;
        lonPrev = lon;
        altPrev = alt;
        // Decrement the Arraylist counter since it will be loading the first
        // element at the start of the loop again.
        ict--;
        // Increment the simulation counter.
        tct++;

        return trajRun;
    }


    //**************************************************************************
    //**************************************************************************
    // GETTERS

    //--------------------------------------------------------------------------
    /**
     * The purpose is obtain the current coordinates of the trajectory object.
     * This method is called by the radar object when performing a track update.
     * 
     * @return The coordinates (latitude, longitude, altitude) of the object at 
     * this point in time.
     */
    public double[] getGEOCoord()
    {
        double[] trajData = new double[3];
        trajData[0] = lat;
        trajData[1] = lon;
        trajData[2] = alt;

        return trajData;
    }

    //**************************************************************************
    //**************************************************************************
    // SETTERS

    //--------------------------------------------------------------------------
    /**
     * The purpose to set the trajectory step resolution.  A high number, such
     * as 10.0, will make the trajectory increments very small, whereas a low
     * number, such as 1.0, will make the steps large.
     * 
     * @param resDial The trajectory resolution value.
     */
    public void setTrajResolution(double resDial)
    {
        this.resDial = resDial;
    }
    
    //--------------------------------------------------------------------------
    /**
     * The purpose to set a 1st trajectory type with specific characteristics - 
     * for demo purposes.
     */
    public void setTrajTypeOne()
    {
        Amp         = 900000.0; // 300 km max altitude
        rangeFact   = (1.0 / resDial) * 2.0 * 3.0 * 0.14; //0.14;
        latRate     = (1.0 / resDial) * 2.0 * 0.15; //0.05;
        lonRate     = (1.0 / resDial) * 2.0 * 0.30; // 0.10
        numPoints   = (int) ( (180.0 / rangeFact) * resDial);
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose to set a 2nd trajectory type with specific characteristics - 
     * for demo purposes.
     */
    public void setTrajTypeTwo()
    {
        Amp         = 900000.0; // 300 km max altitude
        rangeFact   = (1.0 / resDial) * 2.0 * 3.0 * 0.14; //0.14;
        latRate     = (1.0 / resDial) * 0.50 * 0.15; //0.05;
        lonRate     = (1.0 / resDial) * 1.2 * 0.30; // 0.10
        numPoints   = (int) ( (180.0 / rangeFact) * resDial);
    }
    
    //--------------------------------------------------------------------------
    /**
     * The purpose is to use the Path class to build the trajectory curve / line
     * generator - and set its properties.
     * 
     * @param color The color of the trajectory curve / line.
     * @param lineWidth The width of the trajectory curve / line.
     */
    public void setPathObject(Color color, double lineWidth)
    {
        // Build Path object.
        path = new Path();
        usePath = true;

        BasicShapeAttributes attrs = new BasicShapeAttributes();

        attrs.setEnableLighting(true);
        attrs.setOutlineMaterial(new Material(color));
        attrs.setInteriorMaterial(new Material(color));

        attrs.setEnableAntialiasing(true);
        attrs.setOutlineWidth(lineWidth);
//            attrs.setOutlineWidth(2);
        attrs.setInteriorOpacity(1.0);
        attrs.setOutlineOpacity(1.0);
//            attrs.setEnableLighting(true);
        path.setAttributes(attrs);
        path.setDrawVerticals(false);
        path.setVisible(true);
    }
    
    //--------------------------------------------------------------------------
    /**
     * The purpose is to use the Polyline class to build the trajectory 
     * curve / line generator - and set its properties.
     * 
     * @param color The color of the trajectory curve / line.
     * @param lineWidth The width of the trajectory curve / line.
     */
    public void setPolylineObject(Color color, double lineWidth)
    {
        // Build Polyline object.
        polyline = new Polyline();
        usePolyline = true;

        polyline.setColor(color);
        polyline.setLineWidth(lineWidth);
        polyline.setPathType(Polyline.LINEAR);
        polyline.setFollowTerrain(false);
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to set the initial conditions (starting location) of
     * the trajectory object.
     * 
     * @param lat The initial latitude coordinate.
     * @param lon The initial longitude coordinate.
     * @param alt The initial altitude coordinate.
     */
    public void setGeoInit(double lat, double lon, double alt)
    {
        // Current coordinate values.
        this.lat        = lat;
        this.lon        = lon;
        this.alt        = alt;
        // Previous coordinate values.
        this.latPrev    = lat;
        this.lonPrev    = lon;
        this.altPrev    = alt;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to access the Earth model, then to load the trajectory 
     * object into a renderable layer and load that into the Earth model.
     * 
     * @param wwd The World Wind Earth model.
     */
    public void loadWorldWindModel(WorldWindowGLCanvas wwd)
    {
        this.wwd = wwd;

        if( (usePolyline) && (polyline != null) )
        {
            //add layer to WorldWind Earth model
            layer.addRenderable(polyline);
        }
        else if( (usePath) && (path != null) )
        {
            //add layer to WorldWind Earth model
            layer.addRenderable(path);
        }

        this.wwd.getModel().getLayers().add(layer);
    }

} // end of class TrajectoryObjects
