
package earthproj;

/**
 * This class contains all of the user-specified simulation settings and the
 * arrays of trajectory and radar objects.  This is used to transfer all of the
 * data and objects from the driver to the EarthView object.
 * 
 * @author mikescodeprojects
 */
public class ScenarioSettings
{
    // Objects to be used in simulation
    private TrajectoryObjects[] to;
    private RadarObjects[] ro;

    // Earth layer options
    private boolean UTMGridStatus;
    private boolean ATMLayerStatus;
    private boolean StarsLayerStatus;
    private boolean MSVEarthAerialStatus;

    // Initial observer view settings
    private double initViewElevation;
    private double initViewLat;
    private double initViewLon;

    // Earth model clip distance settings
    private boolean setClipDistances;
    private double nearClipDistanceVal;
    private double farClipDistanceVal;
    
    // Simulation delay time value
    private int timeDelay;

    /**
     * Default values are set.
     */
    public ScenarioSettings()
    {
        // Default Earth layer Settings:
        UTMGridStatus           = false;
        ATMLayerStatus          = true;
        StarsLayerStatus        = false;
        MSVEarthAerialStatus    = true;

        // Default clip distance settings.  Note that this will cause "clipping"
        // of the trajectories as they near the far side of the Earth.
        setClipDistances        = false;
        nearClipDistanceVal     = 0.0;
        farClipDistanceVal      = 0.0;

        // Default time delay value
        timeDelay               = 100;
    }

    //--------------------------------------------------------------------------
    // GETTERS

    //--------------------------------------------------------------------------
    /**
     * The purpose is to pass the clip distances to the EarthView object; these 
     * are in turn to passed to the CustomOrbitView object.
     * 
     * @return The user-specified near and far clip distances.
     */
    public double[] getClipDistances()
    {
        double[] clipDistances = new double[2];

        clipDistances[0] = nearClipDistanceVal;
        clipDistances[1] = farClipDistanceVal;

        return clipDistances;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to retrieve the radar objects array - this array contains 
     * all of the radar objects that will be used in the simulation.
     * 
     * @return The array of radar objects.
     */
    public RadarObjects[] getRadarModels()
    {
        return ro;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to retrieve the trajectory objects array - this contains 
     * the trajectory objects that will be used in the simulation.
     * 
     * @return The array of trajectory objects.
     */
    public TrajectoryObjects[] getTrajectoryObjects()
    {
        return to;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to set custom clip distance values.  If return value is 
     * set to true, then the custom clip distance values will be used in the 
     * CustomOrbitView object.
     * 
     * @return Status value (true or false) of the request to set custom 
     * clip distance values.
     */
    public boolean getClipDistanceStatus()
    {
        return setClipDistances;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to pass the user-specified time delay value to the 
     * EarthView object for use in the simulation.
     * 
     * @return The user-specified time delay value.
     */
    public int getTimeDelay()
    {
        return timeDelay;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to get the user-specified visible latitude / longitude 
     * grid pattern layer status - it lets the EarthView object know whether or 
     * not to activate it.
     *
     * @return The boolean status for the UTMGrid layer.
     */
    public boolean getUTMGridOption()
    {
        return UTMGridStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to get the user-specified visible atmosphere layer status 
     * - it lets the EarthView object know whether or not to activate it.
     *
     * @return The boolean status for the atmosphere layer.
     */
    public boolean getATMLayerOption()
    {
        return ATMLayerStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to get the user-specified visible stars layer status - it 
     * lets the EarthView object know whether or not to activate it.
     * 
     * @return The boolean status for the stars layer.
     */
    public boolean getStarsLayerOption()
    {
        return StarsLayerStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to get the user-specified MSV Earth Aerial layer status - 
     * it lets the EarthView object know whether or not to activate it.
     *
     * @return The boolean status for the MSV Earth Aerial layer.
     */
    public boolean getMSVEarthAerialOption()
    {
        return MSVEarthAerialStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to build the [latitude,longitude,altitude] vector which 
     * sets the initial observer's view of the Earth when starting the 
     * simulation.
     * 
     * @return The coordinates of the initial observer's view of the Earth.
     */
    public double[] getViewSettings()
    {
        // Build a 3-element array.
        double[] vect = new double[3];

        // Latitude coordinate of observer's view
        vect[0] = initViewLat;
        // Longitudue coordinate of observer's view
        vect[1] = initViewLon;
        // Altitude value of observer's view
        vect[2] = initViewElevation;

        return vect;
    }

    //--------------------------------------------------------------------------
    // SETTERS

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the radar objects array from the driver - this 
     * will then be retrieved by the EarthView object via another method.
     * 
     * @param ro The array of RadarObjects objects.
     */
    public void loadRadarModels(RadarObjects[] ro)
    {
        this.ro = ro;
    }
    
    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the trajectory objects array from the driver - 
     * this will be then retrieved by the EarthView object via another method.
     * 
     * @param to The array of TrajectoryObjects objects.
     */
    public void loadTrajectoryObjects(TrajectoryObjects[] to)
    {
        this.to = to;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the near and far clip distance values from the 
     * driver - these will be then retrieved by the EarthView object via 
     * another method.
     * 
     * @param nearClipDistanceVal The near clip distance value.
     * @param farClipDistanceVal The far clip distance value.
     */
    public void setClipDistances(double nearClipDistanceVal, 
                                                      double farClipDistanceVal)
    {
        this.nearClipDistanceVal = nearClipDistanceVal;
        this.farClipDistanceVal  = farClipDistanceVal;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the clip distance status from the driver
     * - the status determines whether or not CustomOrbitView will set the 
     * user-specified clip distances.  The status will be retrieved by the 
     * EarthView object via another method.
     * 
     * @param setClipDistances Status (true or false) for using the 
     * user-specified clip distance values.
     */
    public void setClipDistances(boolean setClipDistances)
    {
        this.setClipDistances = setClipDistances;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the time delay value from the driver - this will 
     * be retrieved by the EarthView object via another method.
     * 
     * @param timeDelay The time delay value in milliseconds (integer).
     */
    public void setTimeDelay(int timeDelay)
    {
        this.timeDelay = timeDelay;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the Latitude - Longitude grid pattern layer status 
     * from the driver.  If "true", this status value will be 
     * used by class EarthView to implement the UTMGrid layer in the 
     * Earth model. 
     *
     * @param UTMGridStatus Sets the status (true or false) for the 
     * Earth UTM Grid layer.
     */
    public void setUTMGridOption(boolean UTMGridStatus)
    {
        this.UTMGridStatus = UTMGridStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the atmosphere layer status from the driver.  If 
     * "true", this status value will be used by EarthView to implement the 
     * atmosphere layer. 
     *
     * @param ATMLayerStatus Sets the status (true or false) for the 
     * Earth atmosphere layer.
     */
    public void setATMLayerOption(boolean ATMLayerStatus)
    {
        this.ATMLayerStatus = ATMLayerStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the star layer status from the driver.  If "true", 
     * this status value will be used by EarthView to implement the star layer. 
     *
     * @param StarsLayerStatus Sets the status (true or false) for the 
     * Earth stars layer.
     */
    public void setStarsLayerOption(boolean StarsLayerStatus)
    {
        this.StarsLayerStatus = StarsLayerStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the MSVEarthAerial layer status from the driver.  
     * If "true", this status value will be used by EarthView to implement the 
     * MSVEarthAerial layer. 
     *
     * @param MSVEarthAerialStatus Sets the status (true or false) for the 
     * MSV Earth Aerial layer.
     */
    public void setMSVEarthAerialOption(boolean MSVEarthAerialStatus)
    {
        this.MSVEarthAerialStatus = MSVEarthAerialStatus;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the 3-element vector, which contains the 
     * coordinates of the observer's initial Earth perspective, from the driver.  
     * EarthView will use these values to set up the observer's perspective at 
     * the beginning.
     * 
     * @param vect The 3-element vector which contains the coordinates of
     * the observer's initial Earth perspective.
     */
    public void setViewSettings(double[] vect)
    {
        // Latitude coordinate of observer's view
        initViewLat         = vect[0];
        // Longitudue coordinate of observer's view
        initViewLon         = vect[1];
        // Altitude value of observer's view
        initViewElevation   = vect[2];
    }

} // end of class ScenarioSettings
