package earthproj;

import gov.nasa.worldwind.awt.WorldWindowGLCanvas;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.layers.RenderableLayer;
import gov.nasa.worldwind.render.Polyline;

import java.awt.Color;
import java.util.ArrayList;

/**
 * This class contains all of the properties and behavior (methods) of a 
 * radar object - one which is stationary on the surface of the Earth and tracks
 * a trajectory object.
 *
 * @author mikescodeprojects
 */
public class RadarObjects
{
    // Trajectory object that this radar object will be tracking.
    private TrajectoryObjects to;

    // Radar Object radar line properties
    private final Polyline polyline;
    // ArrayList object to hold the Polyline objects - the ArrayList is loaded
    // into the Earth Layer object in the Earth model.
    private final ArrayList<Position> pos;
    
    // Radar tracking line properties - these are the coordinates that form
    // the radar line which starts at the radar base and ends at the target.
    //      Radar target coordinates.
    private double latTarg;
    private double lonTarg;
    private double altTarg;
    //      Radar base coordinates.
    private double latBase;
    private double lonBase;
    private double altBase;
    // WorldWind Earth model.
    private WorldWindowGLCanvas wwd;
    private final RenderableLayer layer;
    // Vector to hold the trajectory object (to be tracked) location data.
    private double[] targData;
    // Vector from the center of the Earth to the radar base station.
    private Vec4 pointOne;
    // Vector from the radar base station to the TrajectoryObject object 
    // being tracked by the radar.
    private Vec4 pointTwo;
    // Radar tracking slant range limit - if the TrajectoryObject moves beyond 
    // the slant range limit, the radar model will stop tracking it visually.
    private final double rangeLimit;
    // Radar tracking angle limit - if the TrajectoryObject moves outside of 
    // the angle limit, the radar model will stop tracking it visually.
    private final double angleLimit;

    /**
     * The Polyline object is constructed and its properties are set.  The
     * ArrayList object is built which will hold the Position objects that 
     * contain location information of both the radar object and the trajectory
     * object being tracked.  Radar tracking limits - range and angle - are also
     * set here.
     * 
     */
    public RadarObjects()
    {
        // Use the deprecated Polyline class to build the radar line model.
        polyline = new Polyline();
        //      Set the color.
        polyline.setColor(Color.YELLOW);
        //      Set the line width.
        polyline.setLineWidth(1.0);
        polyline.setPathType(Polyline.LINEAR);
        polyline.setFollowTerrain(false);
        //      Only use one segment between start and end points.  This 
        //      prevents polynomial curve fitting.
        polyline.setNumSubsegments(1);
        // The ArrayList object to hold the Polyline object.
        pos = new ArrayList<>();
        // Build the Layer object into which the ArrayList object, carrying the
        // Polyline object, will be loaded.
        layer = new RenderableLayer();
        // Build the target (TrajectoryObject object tracked by the radar model)
        // coordinates vector.
        targData = new double[3];
        // Set the default radar slant range limit.
        rangeLimit = 20_000_000.0;
        // Set the default radar angle limit.
        angleLimit = 90.0;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to show the radar line from the base station to the 
     * target.  This method is called by EarthView during the trajectory 
     * propagation process.
     * 
     * Trajectory Vector: the vector from the center of the Earth to the 
     * coordinates of the trajectory object at a specific point in time.
     * 
     * Radar Base Station Vector: the vector from the center of the Earth to
     * the coordinates of the radar base station.
     * 
     * Radar Range Vector: the vector from the radar base station to the 
     * coordinates of the trajectory object.  The magnitude is the slant range 
     * of the trajectory object as seen by the radar.
     * 
     * Radar Range Angle: the angle between the radar base station Earth surface
     * normal vector and the radar slant range vector.  For example, when this 
     * angle is 90 degrees, the radar range vector would be pointed along the 
     * local horizon.
     */
    public void activateRadarTracking()
    {
        // Class Vec4 which contains the radar slant range vector 
        Vec4 rangeVec;
        // Range vector magnitude
        double rangeMag;
        // Radar slant range angle
        Angle rangeAngle;
        
        // Retrieve the current coordinates of the target trajectory object
        // that the radar object is tracking.
        targData = to.getGEOCoord();
        latTarg = targData[0];  // Latitude
        lonTarg = targData[1];  // Longitude
        altTarg = targData[2];  // Altitude
        
        // Build the vector from the center of the Earth to the 
        // trajectory object.
        pointTwo = wwd.getModel().getGlobe().computePointFromPosition(Position.
                                        fromDegrees(latTarg, lonTarg, altTarg));
        // Compute the slant range vector. 
        // Range vector = trajectory vector - radar base station vector
        rangeVec = pointTwo.subtract3(pointOne);
        // Compute the slant range magnitude
        rangeMag = rangeVec.getLength3();
        // Compute the radar tracking angle
        rangeAngle = pointOne.angleBetween3(rangeVec);
        // If the trajectory object is within the slant range and angle limits
        // of the radar system, then track the object.
        if( rangeMag < rangeLimit && rangeAngle.getDegrees() < angleLimit )
        {    
            // Load the radar object base coordinates into the 1st element of the
            // ArrayList.
            pos.add(0, Position.fromDegrees(latBase, lonBase, altBase));
            // Load the target object coordinates into the 2nd element of the
            // ArrayList.
            pos.add(1, Position.fromDegrees(latTarg, lonTarg, altTarg));
            // Load the ArrayList - which contains the start and end points of the
            // radar tracking line.
            polyline.setPositions(pos);
            // Update (refresh) the Earth model rendering.
            wwd.redrawNow();
            // Clear out the ArrayList.  This will not show on the Earth model 
            // because it's not refreshed until new coordinates are loaded 
            // in the call to this method from EarthView.
            pos.clear();
        }
        // If the radar limits are exceeded then delete 
        // any remaining radar lines.
        else
        {
            // Clear out the ArrayList of any values.
            pos.clear();
            // Load the empty ArrayList into the Polyline object.
            polyline.setPositions(pos);
            // Update (refresh) the Earth model rendering.  This will clear 
            // out any remaining radar lines.
            wwd.redrawNow();
        }
    }

    //--------------------------------------------------------------------------
    //**************************************************************************
    // SETTERS

    //--------------------------------------------------------------------------
    /**
     * The purpose is to set the radar object base coordinates.  This method
     * is called from the driver during the setup process.
     * 
     * @param lat The latitude coordinate of the radar object location.
     * @param lon The longitude coordinate of the radar object location.
     * @param alt The altitude coordinate of the radar object location.
     */
    public void setBaseCoord(double lat, double lon, double alt)
    {
        lonBase = lon;
        latBase = lat;
        altBase = alt;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to: 1) access the Earth model, 2) add the Polyline object
     * to the renderable layer and then add that to the Earth model, 
     * and 3) build the vector from the center of the Earth to the radar base 
     * station (using the Vec4 class).
     * 
     * @param wwd The World Wind Earth model.
     */
    public void loadWorldWindModel(WorldWindowGLCanvas wwd)
    {
        this.wwd = wwd;
        //add layer to WorldWind Earth model
        layer.addRenderable(polyline);
        this.wwd.getModel().getLayers().add(layer);

        // Build the vector from the center of the Earth to the 
        // radar base station.
        pointOne = wwd.getModel().getGlobe().computePointFromPosition(Position.
                                        fromDegrees(latBase, lonBase, altBase));
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the trajectory object which the radar object 
     * will track.
     * 
     * @param to The trajectory object to be tracked.
     */
    public void loadTrajectoryObject(TrajectoryObjects to)
    {
        this.to = to;
    }

} // end of class RadarObjects
