
package earthproj;

import gov.nasa.worldwind.Model;
import gov.nasa.worldwind.View;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.avlist.AVKey;
import gov.nasa.worldwind.awt.WorldWindowGLCanvas;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.layers.LatLonGraticuleLayer;
import gov.nasa.worldwind.layers.Layer;
import gov.nasa.worldwind.view.orbit.OrbitView;
import java.awt.Color;
import java.awt.Dimension;
import javax.swing.JDesktopPane;
import javax.swing.JFrame;
import javax.swing.JInternalFrame;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;

/**
 * The purpose is to:
 * 
 * 1) Build the Swing containers (JFrame, JPanel, etc.) and then 
 * build the WorldWind Earth model and load into one of the JPanel containers.
 * 2) Configure the use-specified Earth model layers - such as UTM grid, 
 * atmosphere, etc.
 * 3) Run a simulation in which the trajectory and radar objects are activated.
 * 
 * @author mikescodeprojects
 */
public final class EarthView
{
    // The JFrame upon which the Earth model is mounted.  Since we're using a
    // JDesktop as the main frame, JInternalFrame is used instead of JFrame.
    // Thus this frame is movable inside the JDesktop frame.
    private final JInternalFrame ew;

    // Main JPanel that's mounted to the JInternalFrame.
    private final JPanel mainPanel;
    private final JPanel subPanel1;

    private final BuildJPanel bp1;
    private final BuildJPanel bp2;

    // WorldWind Earth model canvas.
    private final WorldWindowGLCanvas wwd;
    // A method in this BasicOrbitView subclass is used to

    // ScenarioSettings object which is loaded in the "main" (EarthProj.java)
    // and passed to the EarthView object.
    private final ScenarioSettings ss;

    // The array of trajectory objects that will be propagated 
    // in the simulation.
    private TrajectoryObjects[] to;
    // The array of radar objects that will track specific trajectory objects.
    private RadarObjects[] ro;

    // User-specified initial observer coordinates.
    private final double initViewElevation;
    private final double initViewLat;
    private final double initViewLon;

    /**
     * The ScenarioSettings object is passed to the constructor when the
     * EarthView object is first built in the "main", EarthProj.
     * 
     * The WorldWind Earth model is built and set up here in the constructor.
     * 
     * @param ss The ScenarioSettings object which carries all of the parameters
     * required by the EarthView object.
     */
    public EarthView(ScenarioSettings ss)
    {
        // The ScenarioSettings object is pass through the constructor from the
        // main "EarthProj" - it contains the user-specified parameter settings
        // for the simulation and properties of the simulation objects.
        this.ss = ss;

        //----------------------------------------------------------------------
        // These are the use-specified initial observer coordinates.
        double[] vect;
        vect = ss.getViewSettings();
        // Latitude coordinate of observer's view
        initViewLat         = vect[0];
        // Longitudue coordinate of observer's view
        initViewLon         = vect[1];
        // Altitude value of observer's view
        initViewElevation   = vect[2];
        //----------------------------------------------------------------------

        //----------------------------------------------------------------------
        // Create the main JFrame and Desktop frames
        JFrame mainFSimFrame = new JFrame("Earth View");
        // Set the JFrame size.
        mainFSimFrame.setSize(1640,950);
        mainFSimFrame.setResizable(true);
        mainFSimFrame.setMaximumSize(new Dimension(1640,950));
        // Set so that the simulation will stop when the JFrame is closed
        // manually by the user.
        mainFSimFrame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        // Make the JFrame visible.
        mainFSimFrame.setVisible(true);

        // Create the desktop
        JDesktopPane FSimdesktop = new JDesktopPane();
        // Set the desktop background color using RGB values.
        FSimdesktop.setBackground(new Color(230,230,230)); // 160,1610,160
        FSimdesktop.setVisible(true);
        // Load the desktop into the JFrame
        mainFSimFrame.add(FSimdesktop);

        //**********************************************************************
        // Set up basic JInternalFrame
        //**********************************************************************
        ew = new JInternalFrame("Earth View");
        // Make the JInternalFame re-sizable.  The Earth model size will 
        // remain the same, but the user can manually resize the frame size.
        ew.setResizable(true);
        // Set the size of the JInternalFrame.
        ew.setSize(1150,875);
        ew.setMaximumSize(new Dimension(1150,875));
        // Set location of the JInternalFrame relative to the JDesktop.
        ew.setLocation(220,10);

        //**********************************************************************
        // Set up main JPanel
        //**********************************************************************
        // Set up main JPanel - it will be mounted to the JInternalFrame and 
        // all other JPanels will be mounted to this JPanel.
        int[] fl    = {1, 3, 3};        // Flow control numbers
        int[] dim   = {1150,835};       // Dimensions 1150, 835
        int[] col   = {160,160,160};    // RGB Color numbers

        // Build the custom BuildJPanel object - it contains 
        // the specified JPanel.
        bp1 = new BuildJPanel(fl,dim,col,false,false);
        // Retrieve the JPanel and make it the main JPanel.
        mainPanel = bp1.getJPanel();
        // Add the main JPanel to the JInternalFrame
        ew.add(mainPanel);

        //**********************************************************************
        // Set up sub JPanel - for WorldWind Earth model
        //**********************************************************************
        int[] fl2  = {1, 3, 3};       // Flow control numbers
        int[] dim2 = {1130,835};      // Dimensions
        int[] col2 = {192,192,192};   // RGB Color numbers

        // Build the custom BuildJPanel object - it contains 
        // the specified JPanel.  This JPanel will contain the Earth model.
        bp2 = new BuildJPanel(fl2,dim2,col2,true,true);
        // Retrieve the JPanel and make it the subpanel.
        subPanel1 = bp2.getJPanel();

        // Add the subpanel to the main panel.
        mainPanel.add(subPanel1);

        //**********************************************************************
        // Set up Earth model
        //**********************************************************************
        // Build a WorldWindowGLCanvas object.
        // Note that a WorldWindowsGLCanvas acts like a JPanel object.
        wwd = new WorldWindowGLCanvas();

        // Build a Model object.
        Model m = (Model)WorldWind.createConfigurationComponent
                (AVKey.MODEL_CLASS_NAME);

        // If the user has specified the setting of the near and far clip
        // distance values, then use the CustomOrbitView class to build an 
        // object which extends the BasicOrbitView class and inherits a method
        // of the BasicView class.
        if( ss.getClipDistanceStatus() )
        {
            // Custom work here - this resolves the clip distance issue.
            // Build a CustomOrbitView object which has the clp distances set.
            CustomOrbitView cov = new CustomOrbitView();
            double[] clipDistances = ss.getClipDistances();
            cov.setClipDistValues(clipDistances[0], clipDistances[1]);
            // Load the CustomOrbitView object into the WorldwindowGLCanvas object.
            wwd.setModelAndView(m, (View)cov);
        }
        // Don't set the custom clip distance values.
        else
        {
            wwd.setModel(m);
        }

        //----------------------------------------------------------------------
        // Set the observer's view of the Earth.
        if( initViewElevation > 0.0 )
        {
            OrbitView orbitView = (OrbitView) wwd.getView();
            Position pos = orbitView.getCenterPosition();
            System.out.println("Orbit View Position:");
            System.out.println(pos.elevation+", "+pos.latitude+", "
                    +pos.longitude);
            pos = Position.fromDegrees(initViewLat, initViewLon,
                    initViewElevation);
            orbitView.setCenterPosition(pos);
        }

        // Load WorldWind model into the sub JPanel
        subPanel1.add(wwd);
        // Make the main JInternalFrame visible
        ew.setVisible(true);

        // Add JInternalFrame to JDesktopPane
        FSimdesktop.add(ew,JLayeredPane.MODAL_LAYER);
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to simulate a trajectory with the loaded arrays of 
     * TrajectoryObjects and RadarObjects objects.  Once the last 
     * TrajectoryObjects object indicates that it's not active anymore, the
     * simulation while-loop will cease.
     */
    public void trajSim()
    {
        // Simulation time delay between object trajectory propagation.
        int timeDelay;
        // Number of trajectory objects.
        int numTrajObj;
        // Number of radar objects.
        int numRadObj;
        // Array of trajectory status boolean values - "true" indicates that 
        // that particular trajectory object is active in its trajectory.
        // A value of "false" means that the trajectory is complete (inactive).
        boolean[] trajRun;

        // Retrieve the array of TrajectoryObjects objects.
        to = ss.getTrajectoryObjects();
        // Compute the total number of elements in the array.
        numTrajObj = to.length;
        // Build the boolean array - true while the trajectory is active and 
        // false when the trajectory is complete.
        trajRun = new boolean[numTrajObj];
        // Iterate through the TrajectoryObjects array to load each with the
        // WorldWind model.  Also set the boolean array values to "true" - 
        // which means that the trajectory is active.
        for(int i = 0; i < numTrajObj; i++)
        {
            to[i].loadWorldWindModel(wwd);
            trajRun[i] = true;
        }

        // Retrieve the array of RadarObjects objects.
        ro = ss.getRadarModels();
        // Compute the total number of elements in the array.
        numRadObj = ro.length;
        // Iterate through the RadarObjects array to load each with the
        // WorldWind model.
        for(int i = 0; i < numTrajObj; i++)
        {
            ro[i].loadWorldWindModel(wwd);
        }

        // Retrieve the user-specified time-delay value.
        timeDelay = ss.getTimeDelay();

        // Simulation "while loop" - it operates while the trajectory objects
        // indicate that they are still active in the propagation process.
        while( checkIfAnyTrue(trajRun) )
        {
            // Iterate through the TrajectoryObjects array to command each 
            // trajectory object to propagate itself.
            for(int i = 0; i < numTrajObj; i++)
            {
                // Propagate the object's trajectory.
                if( trajRun[i])
                {    
                    trajRun[i] = to[i].propagateTrajectory();
                }
            }

            // Iterate through the RadarObjects array to command each radar
            // object to update its track.
            for(int i = 0; i < numRadObj; i++)
            {
                ro[i].activateRadarTracking();
            }

            // Delay the simulation for the user-specified amount of time.
            try
            {
                Thread.sleep(timeDelay);
            }
            catch(Exception err)
            {
                System.out.println("Time delay error: "+err+"\n");
            }

        }
        System.out.println("Simulation complete - exiting ...");
    }
    
    //--------------------------------------------------------------------------
    /**
     * The purpose is to cycle through the boolean array and see if any of the
     * elements are "true" - if so then return a value of "true".  This method
     * is called by the trajSim method.
     * 
     * @param queue The array of boolean values that need to be checked to 
     * assess if any of them have a value of "true".
     * 
     * @return A boolean value of "true" if any of the booleans in the array
     * are "true", and "false" if none of the booleans in the array are "true".
     */
    public boolean checkIfAnyTrue(boolean[] queue) 
    {
        // Check the queue for any "true" values before proceeding
        boolean anyTrue = false;
        // Cycle through the boolean array and see if any of the elements
        // are "true" - if so then set the method return value to "true".
        for (int i = 0; i < queue.length; i++) 
        {
            if (queue[i] == true) 
            {
                anyTrue = true;
                break;
            }
        }
        
        return anyTrue;
    }    

    //--------------------------------------------------------------------------
    /**
     * The purpose is to load the user-specified Earth layers 
     * (e.g. atmosphere, stars, etc.) into the WorldWind Earth model.
     */
    public void configureEarthLayers()
    {
        // Possible Layer classes:
        //
        //Stars
        @SuppressWarnings("UnusedAssignment")
        Layer lStars    = null;
        //Atmosphere
        @SuppressWarnings("UnusedAssignment")
        Layer lAtmos    = null;
        //MS Virtual Earth Aerial
        Layer lMSV      = null;

        // Remove the miniature world map
        Layer worldMapLayer = wwd.getModel().getLayers().
                getLayerByName("World Map");
        wwd.getModel().getLayers().remove(worldMapLayer);
        // Remove the compass
        Layer compassLayer = wwd.getModel().getLayers().
                getLayerByName("Compass");
        wwd.getModel().getLayers().remove(compassLayer);

        //**********************************************************************
        // "UTM Grid" Option (NOT embedded by default)
        if(ss.getUTMGridOption())
        {
            Layer graticuleLayer = null;
            try
            {
                graticuleLayer = (Layer) LatLonGraticuleLayer.
                        class.newInstance();
            }
            catch (InstantiationException | IllegalAccessException e)
            {
                System.out.println("Can't get a graticule layer "+e);
            }

            if (graticuleLayer != null)
            {
                graticuleLayer.setEnabled(true);
            }

            wwd.getModel().getLayers().add(graticuleLayer);
        }

        //**********************************************************************
        // "Atmosphere" layer option (embedded and enabled by default)
        if(ss.getATMLayerOption())
        {
            lAtmos = wwd.getModel().getLayers().getLayerByName("Atmosphere");
            if(lAtmos.isEnabled())
                System.out.println("Atmosphere Layer is already Active");
            else
            {
                System.out.println("Atmosphere Layer is NOT Active - "
                        + "activating now.");
                lAtmos.setEnabled(true);
            }
        }
        else
        {
            if( wwd.getModel().getLayers().
                    getLayerByName("Atmosphere") != null )
            {
                System.out.println("Removing Atmosphere layer ...");
                lAtmos = wwd.getModel().getLayers().
                        getLayerByName("Atmosphere");
                wwd.getModel().getLayers().remove(lAtmos);
            }
            else
            {
                System.out.println("Atmosphere Layer not present - "
                        + "no need to remove.");
            }
        }

        //**********************************************************************
        // "Stars" layer option (embedded and enabled by default)
        if(ss.getStarsLayerOption())
        {
            lStars = wwd.getModel().getLayers().getLayerByName("Stars");
            if(lStars.isEnabled())
                System.out.println("Stars Layer is already Active");
            else
            {
                System.out.println("Stars Layer is NOT Active - "
                        + "activating now.");
                lStars.setEnabled(true);
            }
        }
        else
        {
            if( wwd.getModel().getLayers().getLayerByName("Stars") != null )
            {
                System.out.println("Removing Stars layer ...");
                lStars = wwd.getModel().getLayers().getLayerByName("Stars");
                wwd.getModel().getLayers().remove(lStars);
            }
            else
            {
                System.out.println("Star Layer not present - "
                        + "no need to remove.");
            }
        }

        System.out.println("\n\nThe following layers are present in "
                + "the Earth model:\n");
        wwd.getModel().getLayers().forEach(layer ->
        {
            System.out.print(layer.getName()+", ");
            System.out.println("Class = "+layer.getClass().toString());
        });
    }

} // end of class EarthView
