
package earthproj;

import gov.nasa.worldwind.view.orbit.BasicOrbitView;

/**
 * This custom class allows the user to specify the near and far clip distances 
 * to allow the viewing of objects beyond the Earth's horizon (such as 
 * trajectories).
 * 
 * The class hierarchy is as follows:
 * BasicView
 *     -BasicOrbitView
 *            -CustomerOrbitView
 * 
 * BasicOrbitView is derived from BasicView which models the view in terms of a 
 * geographic position, and a pitch, heading, and roll. It provides a mapping 
 * from the geocentric view model to a 3D graphics modelview matrix. BasicView 
 * also manages the projection matrix via a Frustum.
 * 
 * @author mikescodeprojects
 */
public class CustomOrbitView extends BasicOrbitView
{
    // Maximum near clip distance for a perspective projection that avoids 
    // clipping an object at a given distance from the eye point.  The given 
    // distance should specify the smallest distance between the eye and the 
    // object being viewed, but may be an approximation if an exact clip 
    // distance is not required.
    private double nearClipDistanceVal;
    private double farClipDistanceVal;

    /**
     * Constructor #1
     */
    public CustomOrbitView()
    {
        super();
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is for the user to set the near and far clip distances to be 
     * stored in this class.  
     * 
     * @param nCD The "near" clip distance.
     * @param fCD The "far" clip distance.
     */
    public void setClipDistValues(double nCD, double fCD)
    {
        this.nearClipDistanceVal = nCD;
        this.farClipDistanceVal  = fCD;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to set a custom near clip distance so that trajectories 
     * don't get "cut off" on the other side of the Earth.  This method is 
     * inherited from base class BasicView and modified (Overridden).
     * 
     * @return The near clip distance value.
     */
    @Override
    protected double computeNearClipDistance()
    {
        return nearClipDistanceVal;
    }

    //--------------------------------------------------------------------------
    /**
     * The purpose is to set custom far clip distance so that trajectories don't 
     * get "cut off" on the other side of the Earth.  This method is inherited 
     * from base class BasicView and modified (Overridden).
     * 
     * @return The far clip distance value.
     */
    @Override
    protected double computeFarClipDistance()
    {
        return farClipDistanceVal;
    }

} // end of class CustomOrbitView
